﻿/* This client side control is used to display list style result, 
 * such as the result of Hengpi candidate or sentence-level xialian
 * candidiate. In the presentation area, each row conrespond to one 
 * result in the list.
 *
 * Each row contains three part:
 * 1. RadioButton-like cell
 * 2. Candidate text cell
 * 3. An additional hidden field for candidate related data 
 */

// Register namespace
Type.registerNamespace("View");

// CandidateList Constructor
View.CandidateList = function(element) {

    // This CandidateList control should bind to a <div>
    // element in DOM tree.
    View.CandidateList.initializeBase(this, [element]);
    
    // Group size is to determine when to insert a empty line
    // into the list view. E.g.if the m_iGroupSize is 10 which
    // means a empty line will be inserted after every 10 result.
    // This value should be greater than 0. The default value is 10.
    this.m_iGroupSize = 10;
    
    // This const prefix is used to build a string id for each element
    // which is dynamically created during the process of construct view
    this.m_gsIdPrefixTable = "Table";
    this.m_gsIdPrefixTBody = "TBody";
    this.m_gsIdPrefixCandidate = "Candidate";
    this.m_gsIdPrefixRadioSelected = "RadioSelected";
    this.m_gsIdPrefixRadioUnSelected = "RadioUnSelected";
    
    // This field is to indicate that what purpose for current CandidateList
    // Control instance. Generally, this field will be filled with string: "Hengpi"
    // or "XialianNormal".
    // This field will also be used as part of the id of those dynamically generated
    // element within the control.
    this.m_sPurpose;
    
    // Delegates for CandidateList related event handler
    // Click handler is invoked when a row of view is clicked.
    this.m_dgClickDelegate = null;
    // Hover handler is invoked when mouse hover on a row.
    this.m_dgHoverDelegate = null;
    // Unhover handler is invoked when mouse leaves a row.
    this.m_dgUnhoverDelegate = null;
    
    // This two field store the current user selected candidate index
    // and last selected candidate index.
    this.m_iOldCandidateId = -1;
    this.m_iNewCandidateId = -1;
    
   
}

// Candidate prototype implementation
View.CandidateList.prototype = 
{
    // Properties
    set_purpose: function(purpose)
    {
        this.m_sPurpose = purpose;
    },
    
    get_purpose: function()
    {
        return this.m_sPurpose;
    },
    
    set_groupSize: function(iGroupSize)
    {
        if(iGroupSize <= 0)
        {
            Sys.Debug.fail("Group size should be greater than zero.");
        }
        else
        {
            this.m_iGroupSize = iGroupSize;
        }
    },
    
    get_groupSize: function()
    {
        return this.m_iGroupSize;
    },
    
    // Event handlers for click event
    add_click: function(handler) 
    {
        this.get_events().addHandler("click", handler);
    },
    
    remove_click: function(handler) 
    {
        this.get_events().removeHandler("click", handler);
    },

    // Event handlers for hover event
    add_hover: function(handler) 
    {
        this.get_events().addHandler("hover", handler);
    },
    
    remove_hover: function(handler) 
    {
        this.get_events().removeHandler("hover", handler);
    },

    // Event handlers for unhover event
    add_unhover: function(handler) 
    {
        this.get_events().addHandler("unhover", handler);
    },
    
    remove_unhover: function(handler) 
    {
        this.get_events().removeHandler("unhover", handler);
    },
    
    // This initialze function will be called by the runtime when
    // this control is created. Here we create event delegete 
    // in initialize funtion. Then construct some basic DOM element
    // needed in this control.
    initialize: function() 
    {
        // Under the element who bind to current control, we append a
        // table to the element.
        var tbl = document.createElement("table");
        tbl.id = this.m_sPurpose + this.m_gsIdPrefixTable;
        
        var tbd = document.createElement("tbody");
        tbd.id = this.m_sPurpose + this.m_gsIdPrefixTBody;
        tbl.appendChild(tbd);
        this.get_element().appendChild(tbl);

        // Create delegates used by this control
        if (this.m_dgClickDelegate === null) {
            this.m_dgClickDelegate = Function.createDelegate(this, this._clickHandler);
        }

        if (this.m_dgHoverDelegate === null) {
            this.m_dgHoverDelegate = Function.createDelegate(this, this._hoverHandler);
        }

        if (this.m_dgUnhoverDelegate === null) {
            this.m_dgUnhoverDelegate = Function.createDelegate(this, this._unhoverHandler);
        }
        
        View.CandidateList.callBaseMethod(this, "initialize");

    },
    
    // These three internal defined handlers are used to 
    // create corresponding delegates. In this pattern, we
    // can decouple the control initialization logic from client
    // custom event handler assignment.
    _clickHandler: function(event) 
    {
        var h = this.get_events().getHandler("click");
        if (h)
        {
            var sender = this._getSender(event);
            while(sender.tagName != "TR")
	        {
	            sender = sender.parentNode;
	        }
            var iNewSelectId = sender.id.slice(sender.id.lastIndexOf("_")+1);
            var sCandidate = this.setSelectedCandidateId(parseInt(iNewSelectId));
            var iSegmentPatternIndex = sender.childNodes[1].lastChild.value;
            var rgArguments = new Array();
            rgArguments[0] = iNewSelectId;
            rgArguments[1] = sCandidate;
            rgArguments[2] = iSegmentPatternIndex;
            h(sender, rgArguments);
        }
    },

    _hoverHandler: function(event) 
    {
        var h = this.get_events().getHandler("hover");
        var sender = this._getSender(event);
        if (h) h(sender, Sys.EventArgs.Empty);
    },

    _unhoverHandler: function(event) 
    {
        var h = this.get_events().getHandler("unhover");
        var sender = this._getSender(event);
        if (h) h(sender, Sys.EventArgs.Empty);
    },


    // Release resources before control is disposed. The main work here
    // is to remove the handler from DOM element and delete the 
    // event delegates created during intializtion phase.
    dispose: function() 
    {
        var tbd = $get(this.m_sPurpose + this.m_gsIdPrefixTBody);
        if(!tbd)
        {
            Sys.Debug.fail("The internal table was not created successfully.");
            return;
        }
        
        var iRowCount = tbd.childNodes.length;
        if (this.m_dgClickDelegate)
        {
            for( var i=0;i<iRowCount;i++)
            {
                if(tbd.childNodes[i].id != "")
                {
                    Sys.UI.DomEvent.removeHandler(tbd.childNodes[i], "click", this.m_dgClickDelegate);
                }
            }
            delete this.m_dgClickDelegate;
        }

        if (this.m_dgHoverDelegate) 
        {
            for( var i=0;i<iRowCount;i++)
            {
                if(tbd.childNodes[i].id != "")
                {
                    Sys.UI.DomEvent.removeHandler(tbd.childNodes[i], "mouseover", this.m_dgHoverDelegate);
                }
            }
            delete this.m_dgHoverDelegate;
        }

        if (this.m_dgUnhoverDelegate) 
        {
            for( var i=0;i<iRowCount;i++)
            {
                if(tbd.childNodes[i].id != "")
                {
                    Sys.UI.DomEvent.removeHandler(tbd.childNodes[i], "mouseout", this.m_dgUnhoverDelegate);
                }
            }
            delete this.m_dgUnhoverDelegate;
        }
        
        View.CandidateList.callBaseMethod(this, "dispose");
    },

    /// <summary>
	/// This method is exposed for adding a candidate result to
    /// current list view.
	/// </summary>
    /// <accessibility>public</accessibility>
	/// <param name="sCandidate">A candidate string</param>
	/// <param name="hidValue">The extra data associated to current candidate string</param>
    addCandidate: function(sCandidate, hidValue)
    {
        var tbd = $get(this.m_sPurpose + this.m_gsIdPrefixTBody);
        if(!tbd)
        {
            Sys.Debug.fail("The internal table was not created successfully.");
            return;
        }
        
        // Get the total number of candidates currently in the list.
        // This number will be used to form the id of newly created
        // DOM element.
        var iCurrentCandidateCount = this._getCandidateCount();
        
        // Create a new row for current candidate result.
        var trcandidate = document.createElement("tr");
        trcandidate.id = this.m_sPurpose + this.m_gsIdPrefixCandidate + "_" + iCurrentCandidateCount;
        

        // 1. The first cell(td) is to store candidate information
        // Create a new candidate cell which contains the candidate string and associated data.        
        var tdcandidate = document.createElement("td");
        // The CSS style name is programatically assigned.
        tdcandidate.className = this.m_sPurpose + this.m_gsIdPrefixCandidate;
        
        // Create a text node for candidate string and put it into the cell.
        var txtcandidate = document.createTextNode(sCandidate);
        tdcandidate.appendChild(txtcandidate); 
        
        // Create a hidden field to store the candidate associated value.
        // Actually, this field is only useful if current list is use to 
        // display Xialian sentence-level candidates, in which case, this 
        // field is to store the sentence segment pattern of current 
        // Xialian candidate string.
        var hidsegment = document.createElement("input");
        hidsegment.name = "hdValue";
        hidsegment.type = "hidden";
        hidsegment.value = hidValue;
        // This field is also appended to the candidate cell.
        tdcandidate.appendChild(hidsegment); 
        
        // 2. The second cell(td) is to fill a radiobutton-like elements
        var tdradio =  document.createElement("td");
        tdradio.className = "radioTD";
        //tdradio.style.width = "18px";
        
        //alert(tdradio.style.width);
        // 2.1 Create a element which is shown when current candidate is selected.
        var divselected = document.createElement("div");
        divselected.className = "radioselected";
        divselected.style.display = "none";
        divselected.id = this.m_sPurpose + this.m_gsIdPrefixRadioSelected + iCurrentCandidateCount;
        
        // 2.2 Create a element which is shown when current candidate is unselected.
        var divsunelected = document.createElement("div");
        divsunelected.className ="radiounselected";
        divsunelected.style.display = "";
        divsunelected.id = this.m_sPurpose + this.m_gsIdPrefixRadioUnSelected + iCurrentCandidateCount;
        
        // Append these two elements to the second cell.
        tdradio.appendChild(divselected);
        tdradio.appendChild(divsunelected);
        
        // Add event handler to current row element(tr)
        if (this.m_dgClickDelegate === null) {
            this.m_dgClickDelegate = Function.createDelegate(this, this._clickHandler);
        }
        Sys.UI.DomEvent.addHandler(trcandidate, 'click', this.m_dgClickDelegate);
        
        if (this.m_dgHoverDelegate === null) {
            this.m_dgHoverDelegate = Function.createDelegate(this, this._hoverHandler);
        }
        Sys.UI.DomEvent.addHandler(trcandidate, 'mouseover', this.m_dgHoverDelegate);
        
        if (this.m_dgUnhoverDelegate === null) {
            this.m_dgUnhoverDelegate = Function.createDelegate(this, this._unhoverHandler);
        }
        Sys.UI.DomEvent.addHandler(trcandidate, 'mouseout', this.m_dgUnhoverDelegate);
        
        // Append the two cells to current candidate row
        trcandidate.appendChild(tdradio);
        trcandidate.appendChild(tdcandidate);
        // Append current candidate row to <table>
        tbd.appendChild(trcandidate);
        
        // Check if we should insert the empty line to list view
        if( (iCurrentCandidateCount+1) % this.m_iGroupSize == 0) 
        {
            var trempty = document.createElement("tr");
            trempty.className = "emptyrow";
            tbd.appendChild(trempty);
        }
    },
    
    /// <summary>
	/// This method is to remove all the candidate result from list
	/// </summary>
    /// <accessibility>public</accessibility>
    clear: function()
    {
        var tbd = this._getInnerTable();
        var iCurrentCandidateCount = tbd.childNodes.length;
        for(var i=0; i<iCurrentCandidateCount; i++)
        {
            tbd.removeChild(tbd.childNodes[0]);
        }
    },
    
    /// <summary>
	/// This method is to set newly selected candidate by candidate idx
    /// in current list view. The valid idx range is from -1 to (candidate count -1).
    /// So -1 is a special value to indicate that unselected all the candidates.
	/// </summary>
    /// <accessibility>public</accessibility>
	/// <param name="iSelectedCandidateId">Index of current selected candidate</param>
	/// <return value>If success, it will return current selected candidate</return value>
    setSelectedCandidateId: function(iSelectedCandidateId)
    {
        var iCandidateCount = this._getCandidateCount();
        if(iSelectedCandidateId < -1 || iSelectedCandidateId >= iCandidateCount)
        {
            Sys.Debug.fail("Selected candidate index is out of range. iSelectedCandidateId=" + iSelectedCandidateId);
        }
        
        this.m_iOldCandidateId = this.m_iNewCandidateId;
        this.m_iNewCandidateId = iSelectedCandidateId;
        
        // Unselect the old radio
        if(this.m_iOldCandidateId > -1)
        {
            var oCandidateItem = $get(this.m_sPurpose + this.m_gsIdPrefixRadioSelected + this.m_iOldCandidateId);
            if(oCandidateItem)
            {
                oCandidateItem.style.display = "none";
            }
            
            oCandidateItem = $get(this.m_sPurpose + this.m_gsIdPrefixRadioUnSelected + this.m_iOldCandidateId);
            if(oCandidateItem)
            {
                oCandidateItem.style.display = "";
            }
            
        }
        
        if(-1 != this.m_iNewCandidateId)
        {
            // Select the new radio
            var oCandidateItem = $get(this.m_sPurpose + this.m_gsIdPrefixRadioSelected + this.m_iNewCandidateId);
            if(oCandidateItem)
            {
                oCandidateItem.style.display = "";
            }
            
            oCandidateItem = $get(this.m_sPurpose + this.m_gsIdPrefixRadioUnSelected + this.m_iNewCandidateId);
            if(oCandidateItem)
            {
                oCandidateItem.style.display = "none";
            }
            return this._getCandidateById(this.m_iNewCandidateId);
        }
    },
    
    /// <summary>
	/// This method is to get candidate count in current list
	/// </summary>
    /// <accessibility>private</accessibility>
	/// <return value>If success, it will return the number of candidate in list</return value>
    _getCandidateCount: function()
    {
        var tbd = this._getInnerTable();
        var iCount = 0;
        for( var i=0; i<tbd.childNodes.length; i++)
        {
            // Each child node is a candidate row in the list.
            // The node whose id is "" is the empty row.
            if(tbd.childNodes[i].id != "")
            {
                iCount++;
            }
        }
        return iCount;
    },
    
    /// <summary>
	/// This method is to get internal table who contains the candidate result
	/// </summary>
    /// <accessibility>private</accessibility>
	/// <return value>If success, it will return the inner table element</return value>
    _getInnerTable: function()
    {
        var tbd = $get(this.m_sPurpose + this.m_gsIdPrefixTBody);
        if(!tbd)
        {
            Sys.Debug.fail("The internal table was not created successfully.");
        }
        return tbd;
    },
    
    /// <summary>
	/// This method is to get the element who invoked current evnet.
	/// </summary>
    /// <accessibility>private</accessibility>
	/// <param name="evt">event object</param>
	/// <return value>If success, it will return DOM element</return value>
    _getSender: function(evt)
    {
        var target = evt.target;              // DOM standard event model
        if (!target)
        {
            target = evt.srcElement;          // IE event model
        }
        return target;
    },
    
    /// <summary>
	/// This method is to get candidate by index
	/// </summary>
	/// <param name="iCandidateId">Index of current selected candidate</param>
    /// <accessibility>private</accessibility>
    _getCandidateById: function(iCandidateId)
    {
        var sSelectedCandidate;
	    if(document.all)
	    {   
	        sSelectedCandidate = $get(this.m_sPurpose + this.m_gsIdPrefixCandidate + "_" + iCandidateId).lastChild.innerText;
	    }
	    else
	    {
	        sSelectedCandidate = $get(this.m_sPurpose + this.m_gsIdPrefixCandidate + "_" + iCandidateId).lastChild.textContent;
	    }
	    return sSelectedCandidate;
    }
    
}
View.CandidateList.registerClass("View.CandidateList", Sys.UI.Control);

// Since this script is not loaded by System.Web.Handlers.ScriptResourceHandler
// invoke Sys.Application.notifyScriptLoaded to notify ScriptManager 
// that this is the end of the script.
if (typeof(Sys) !== "undefined") Sys.Application.notifyScriptLoaded();
